﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/observabilityadmin/ObservabilityAdminRequest.h>
#include <aws/observabilityadmin/ObservabilityAdmin_EXPORTS.h>
#include <aws/observabilityadmin/model/ResourceType.h>
#include <aws/observabilityadmin/model/TelemetryState.h>
#include <aws/observabilityadmin/model/TelemetryType.h>

#include <utility>

namespace Aws {
namespace ObservabilityAdmin {
namespace Model {

/**
 */
class ListResourceTelemetryRequest : public ObservabilityAdminRequest {
 public:
  AWS_OBSERVABILITYADMIN_API ListResourceTelemetryRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListResourceTelemetry"; }

  AWS_OBSERVABILITYADMIN_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p> A string used to filter resources which have a
   * <code>ResourceIdentifier</code> starting with the
   * <code>ResourceIdentifierPrefix</code>. </p>
   */
  inline const Aws::String& GetResourceIdentifierPrefix() const { return m_resourceIdentifierPrefix; }
  inline bool ResourceIdentifierPrefixHasBeenSet() const { return m_resourceIdentifierPrefixHasBeenSet; }
  template <typename ResourceIdentifierPrefixT = Aws::String>
  void SetResourceIdentifierPrefix(ResourceIdentifierPrefixT&& value) {
    m_resourceIdentifierPrefixHasBeenSet = true;
    m_resourceIdentifierPrefix = std::forward<ResourceIdentifierPrefixT>(value);
  }
  template <typename ResourceIdentifierPrefixT = Aws::String>
  ListResourceTelemetryRequest& WithResourceIdentifierPrefix(ResourceIdentifierPrefixT&& value) {
    SetResourceIdentifierPrefix(std::forward<ResourceIdentifierPrefixT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> A list of resource types used to filter resources supported by telemetry
   * config. If this parameter is provided, the resources will be returned in the
   * same order used in the request. </p>
   */
  inline const Aws::Vector<ResourceType>& GetResourceTypes() const { return m_resourceTypes; }
  inline bool ResourceTypesHasBeenSet() const { return m_resourceTypesHasBeenSet; }
  template <typename ResourceTypesT = Aws::Vector<ResourceType>>
  void SetResourceTypes(ResourceTypesT&& value) {
    m_resourceTypesHasBeenSet = true;
    m_resourceTypes = std::forward<ResourceTypesT>(value);
  }
  template <typename ResourceTypesT = Aws::Vector<ResourceType>>
  ListResourceTelemetryRequest& WithResourceTypes(ResourceTypesT&& value) {
    SetResourceTypes(std::forward<ResourceTypesT>(value));
    return *this;
  }
  inline ListResourceTelemetryRequest& AddResourceTypes(ResourceType value) {
    m_resourceTypesHasBeenSet = true;
    m_resourceTypes.push_back(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> A key-value pair to filter resources based on the telemetry type and the
   * state of the telemetry configuration. The key is the telemetry type and the
   * value is the state. </p>
   */
  inline const Aws::Map<TelemetryType, TelemetryState>& GetTelemetryConfigurationState() const { return m_telemetryConfigurationState; }
  inline bool TelemetryConfigurationStateHasBeenSet() const { return m_telemetryConfigurationStateHasBeenSet; }
  template <typename TelemetryConfigurationStateT = Aws::Map<TelemetryType, TelemetryState>>
  void SetTelemetryConfigurationState(TelemetryConfigurationStateT&& value) {
    m_telemetryConfigurationStateHasBeenSet = true;
    m_telemetryConfigurationState = std::forward<TelemetryConfigurationStateT>(value);
  }
  template <typename TelemetryConfigurationStateT = Aws::Map<TelemetryType, TelemetryState>>
  ListResourceTelemetryRequest& WithTelemetryConfigurationState(TelemetryConfigurationStateT&& value) {
    SetTelemetryConfigurationState(std::forward<TelemetryConfigurationStateT>(value));
    return *this;
  }
  inline ListResourceTelemetryRequest& AddTelemetryConfigurationState(TelemetryType key, TelemetryState value) {
    m_telemetryConfigurationStateHasBeenSet = true;
    m_telemetryConfigurationState.emplace(key, value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> A key-value pair to filter resources based on tags associated with the
   * resource. For more information about tags, see <a
   * href="https://docs.aws.amazon.com/whitepapers/latest/tagging-best-practices/what-are-tags.html">What
   * are tags?</a> </p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetResourceTags() const { return m_resourceTags; }
  inline bool ResourceTagsHasBeenSet() const { return m_resourceTagsHasBeenSet; }
  template <typename ResourceTagsT = Aws::Map<Aws::String, Aws::String>>
  void SetResourceTags(ResourceTagsT&& value) {
    m_resourceTagsHasBeenSet = true;
    m_resourceTags = std::forward<ResourceTagsT>(value);
  }
  template <typename ResourceTagsT = Aws::Map<Aws::String, Aws::String>>
  ListResourceTelemetryRequest& WithResourceTags(ResourceTagsT&& value) {
    SetResourceTags(std::forward<ResourceTagsT>(value));
    return *this;
  }
  template <typename ResourceTagsKeyT = Aws::String, typename ResourceTagsValueT = Aws::String>
  ListResourceTelemetryRequest& AddResourceTags(ResourceTagsKeyT&& key, ResourceTagsValueT&& value) {
    m_resourceTagsHasBeenSet = true;
    m_resourceTags.emplace(std::forward<ResourceTagsKeyT>(key), std::forward<ResourceTagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> A number field used to limit the number of results within the returned list.
   * </p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListResourceTelemetryRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The token for the next set of items to return. A previous call generates
   * this token. </p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListResourceTelemetryRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_resourceIdentifierPrefix;

  Aws::Vector<ResourceType> m_resourceTypes;

  Aws::Map<TelemetryType, TelemetryState> m_telemetryConfigurationState;

  Aws::Map<Aws::String, Aws::String> m_resourceTags;

  int m_maxResults{0};

  Aws::String m_nextToken;
  bool m_resourceIdentifierPrefixHasBeenSet = false;
  bool m_resourceTypesHasBeenSet = false;
  bool m_telemetryConfigurationStateHasBeenSet = false;
  bool m_resourceTagsHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace ObservabilityAdmin
}  // namespace Aws
