/*
 * Copyright 2022 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.api.internal.attributes;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Sets;
import org.gradle.api.Named;
import org.gradle.api.attributes.Attribute;
import org.gradle.api.attributes.AttributeContainer;
import org.gradle.api.provider.Provider;
import org.jspecify.annotations.Nullable;

import java.util.Comparator;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.TreeMap;

/**
 * Joins a primary and fallback attribute container to each other. Any attribute in the primary
 * container will override attributes in the fallback container. All mutation operations are
 * forwarded to the primary container.
 */
/* package */ final class HierarchicalMutableAttributeContainer extends AbstractAttributeContainer {
    private final AttributesFactory attributesFactory;
    private final AttributeContainerInternal fallback;
    private final AttributeContainerInternal primary;

    HierarchicalMutableAttributeContainer(AttributesFactory attributesFactory, AttributeContainerInternal fallback, AttributeContainerInternal primary) {
        this.attributesFactory = attributesFactory;
        this.fallback = fallback;
        this.primary = primary;
    }

    @Override
    public Set<Attribute<?>> keySet() {
        return Sets.union(fallback.keySet(), primary.keySet());
    }

    @Override
    public <T> AttributeContainer attribute(Attribute<T> key, T value) {
        primary.attribute(key, value);
        return this;
    }

    @Override
    public <T> AttributeContainer attributeProvider(Attribute<T> key, Provider<? extends T> provider) {
        primary.attributeProvider(key, provider);
        return this;
    }

    @Override
    public AttributeContainer addAllLater(AttributeContainer other) {
        primary.addAllLater(other);
        return this;
    }

    @Nullable
    @Override
    public <T> T getAttribute(Attribute<T> key) {
        if (!isValidAttributeRequest(key)) {
            return null;
        }

        T attribute = primary.getAttribute(key);
        if (attribute != null) {
            return attribute;
        }
        return fallback.getAttribute(key);
    }

    @Override
    public Provider<Map<Attribute<?>, AttributeEntry<?>>> getEntriesProvider() {
        return primary.getEntriesProvider().zip(fallback.getEntriesProvider(), (prim, fall) ->
            ImmutableMap.<Attribute<?>, AttributeEntry<?>>builderWithExpectedSize(prim.size() + fall.size())
                .putAll(fall)
                .putAll(prim)
                .buildKeepingLast()
        );
    }

    @Override
    public ImmutableAttributes asImmutable() {
        if (primary.isEmpty()) {
            return fallback.asImmutable();
        }
        if (fallback.isEmpty()) {
            return primary.asImmutable();
        }
        return attributesFactory.concat(fallback.asImmutable(), primary.asImmutable());
    }

    @Override
    public <T extends Named> T named(Class<T> type, String name) {
        return primary.named(type, name);
    }

    @Override
    public boolean equals(@Nullable Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        HierarchicalMutableAttributeContainer that = (HierarchicalMutableAttributeContainer) o;
        return fallback.equals(that.fallback) && primary.equals(that.primary);
    }

    @Override
    public int hashCode() {
        return Objects.hash(fallback, primary);
    }

    @Override
    public String toString() {
        final Map<Attribute<?>, Object> sorted = new TreeMap<>(Comparator.comparing(Attribute::getName));
        fallback.keySet().forEach(key -> sorted.put(key, fallback.getAttribute(key)));
        primary.keySet().forEach(key -> sorted.put(key, primary.getAttribute(key)));
        return sorted.toString();
    }
}
