/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.cache;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.Weigher;
import java.io.IOException;
import java.util.Iterator;
import org.apache.cassandra.cache.ICache;
import org.apache.cassandra.cache.RefCountedMemory;
import org.apache.cassandra.concurrent.ImmediateExecutor;
import org.apache.cassandra.io.ISerializer;
import org.apache.cassandra.io.util.MemoryInputStream;
import org.apache.cassandra.io.util.MemoryOutputStream;
import org.apache.cassandra.io.util.WrappedDataOutputStreamPlus;
import org.apache.cassandra.utils.FBUtilities;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SerializingCache<K, V>
implements ICache<K, V> {
    private static final Logger logger = LoggerFactory.getLogger(SerializingCache.class);
    private final Cache<K, RefCountedMemory> cache;
    private final ISerializer<V> serializer;

    private SerializingCache(long capacity, Weigher<K, RefCountedMemory> weigher, ISerializer<V> serializer) {
        this.serializer = serializer;
        this.cache = Caffeine.newBuilder().weigher(weigher).maximumWeight(capacity).executor(ImmediateExecutor.INSTANCE).removalListener((key, mem, cause) -> {
            if (cause.wasEvicted()) {
                mem.unreference();
            }
        }).build();
    }

    public static <K, V> SerializingCache<K, V> create(long weightedCapacity, Weigher<K, RefCountedMemory> weigher, ISerializer<V> serializer) {
        return new SerializingCache<K, V>(weightedCapacity, weigher, serializer);
    }

    public static <K, V> SerializingCache<K, V> create(long weightedCapacity, ISerializer<V> serializer) {
        return SerializingCache.create(weightedCapacity, (key, value) -> {
            long size = value.size();
            if (size > Integer.MAX_VALUE) {
                throw new IllegalArgumentException("Serialized size must not be more than 2GiB");
            }
            return (int)size;
        }, serializer);
    }

    private V deserialize(RefCountedMemory mem) {
        try {
            return this.serializer.deserialize(new MemoryInputStream(mem));
        }
        catch (IOException e) {
            logger.trace("Cannot fetch in memory data, we will fallback to read from disk ", (Throwable)e);
            return null;
        }
    }

    private RefCountedMemory serialize(V value) {
        RefCountedMemory freeableMemory;
        long serializedSize = this.serializer.serializedSize(value);
        if (serializedSize > Integer.MAX_VALUE) {
            throw new IllegalArgumentException(String.format("Unable to allocate %s", FBUtilities.prettyPrintMemory(serializedSize)));
        }
        try {
            freeableMemory = new RefCountedMemory(serializedSize);
        }
        catch (OutOfMemoryError e) {
            return null;
        }
        try {
            this.serializer.serialize(value, new WrappedDataOutputStreamPlus(new MemoryOutputStream(freeableMemory)));
        }
        catch (IOException e) {
            freeableMemory.unreference();
            throw new RuntimeException(e);
        }
        return freeableMemory;
    }

    @Override
    public long capacity() {
        return this.cache.policy().eviction().get().getMaximum();
    }

    @Override
    public void setCapacity(long capacity) {
        this.cache.policy().eviction().get().setMaximum(capacity);
    }

    public boolean isEmpty() {
        return this.cache.asMap().isEmpty();
    }

    @Override
    public int size() {
        return this.cache.asMap().size();
    }

    @Override
    public long weightedSize() {
        return this.cache.policy().eviction().get().weightedSize().getAsLong();
    }

    @Override
    public void clear() {
        this.cache.invalidateAll();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public V get(K key) {
        RefCountedMemory mem = this.cache.getIfPresent(key);
        if (mem == null) {
            return null;
        }
        if (!mem.reference()) {
            return null;
        }
        try {
            V v = this.deserialize(mem);
            return v;
        }
        finally {
            mem.unreference();
        }
    }

    @Override
    public void put(K key, V value) {
        RefCountedMemory old;
        RefCountedMemory mem = this.serialize(value);
        if (mem == null) {
            return;
        }
        try {
            old = this.cache.asMap().put(key, mem);
        }
        catch (Throwable t2) {
            mem.unreference();
            throw t2;
        }
        if (old != null) {
            old.unreference();
        }
    }

    @Override
    public boolean putIfAbsent(K key, V value) {
        RefCountedMemory old;
        RefCountedMemory mem = this.serialize(value);
        if (mem == null) {
            return false;
        }
        try {
            old = this.cache.asMap().putIfAbsent(key, mem);
        }
        catch (Throwable t2) {
            mem.unreference();
            throw t2;
        }
        if (old != null) {
            mem.unreference();
        }
        return old == null;
    }

    @Override
    public boolean replace(K key, V oldToReplace, V value) {
        boolean success;
        RefCountedMemory old = this.cache.getIfPresent(key);
        if (old == null) {
            return false;
        }
        if (!old.reference()) {
            return false;
        }
        V oldValue = this.deserialize(old);
        old.unreference();
        if (!oldValue.equals(oldToReplace)) {
            return false;
        }
        RefCountedMemory mem = this.serialize(value);
        if (mem == null) {
            return false;
        }
        try {
            success = this.cache.asMap().replace(key, old, mem);
        }
        catch (Throwable t2) {
            mem.unreference();
            throw t2;
        }
        if (success) {
            old.unreference();
        } else {
            mem.unreference();
        }
        return success;
    }

    @Override
    public void remove(K key) {
        RefCountedMemory mem = (RefCountedMemory)this.cache.asMap().remove(key);
        if (mem != null) {
            mem.unreference();
        }
    }

    @Override
    public Iterator<K> keyIterator() {
        return this.cache.asMap().keySet().iterator();
    }

    @Override
    public Iterator<K> hotKeyIterator(int n) {
        return this.cache.policy().eviction().get().hottest(n).keySet().iterator();
    }

    @Override
    public boolean containsKey(K key) {
        return this.cache.asMap().containsKey(key);
    }
}

