﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/inspector2/Inspector2Request.h>
#include <aws/inspector2/Inspector2_EXPORTS.h>
#include <aws/inspector2/model/FilterCriteria.h>
#include <aws/inspector2/model/SortCriteria.h>

#include <utility>

namespace Aws {
namespace Inspector2 {
namespace Model {

/**
 */
class ListFindingsRequest : public Inspector2Request {
 public:
  AWS_INSPECTOR2_API ListFindingsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListFindings"; }

  AWS_INSPECTOR2_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The maximum number of results the response can return. If your request would
   * return more than the maximum the response will return a <code>nextToken</code>
   * value, use this value when you call the action again to get the remaining
   * results.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListFindingsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A token to use for paginating results that are returned in the response. Set
   * the value of this parameter to null for the first request to a list action. If
   * your response returns more than the <code>maxResults</code> maximum value it
   * will also return a <code>nextToken</code> value. For subsequent calls, use the
   * <code>nextToken</code> value returned from the previous request to continue
   * listing results after the first page.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListFindingsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Details on the filters to apply to your finding results.</p>
   */
  inline const FilterCriteria& GetFilterCriteria() const { return m_filterCriteria; }
  inline bool FilterCriteriaHasBeenSet() const { return m_filterCriteriaHasBeenSet; }
  template <typename FilterCriteriaT = FilterCriteria>
  void SetFilterCriteria(FilterCriteriaT&& value) {
    m_filterCriteriaHasBeenSet = true;
    m_filterCriteria = std::forward<FilterCriteriaT>(value);
  }
  template <typename FilterCriteriaT = FilterCriteria>
  ListFindingsRequest& WithFilterCriteria(FilterCriteriaT&& value) {
    SetFilterCriteria(std::forward<FilterCriteriaT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Details on the sort criteria to apply to your finding results.</p>
   */
  inline const SortCriteria& GetSortCriteria() const { return m_sortCriteria; }
  inline bool SortCriteriaHasBeenSet() const { return m_sortCriteriaHasBeenSet; }
  template <typename SortCriteriaT = SortCriteria>
  void SetSortCriteria(SortCriteriaT&& value) {
    m_sortCriteriaHasBeenSet = true;
    m_sortCriteria = std::forward<SortCriteriaT>(value);
  }
  template <typename SortCriteriaT = SortCriteria>
  ListFindingsRequest& WithSortCriteria(SortCriteriaT&& value) {
    SetSortCriteria(std::forward<SortCriteriaT>(value));
    return *this;
  }
  ///@}
 private:
  int m_maxResults{0};

  Aws::String m_nextToken;

  FilterCriteria m_filterCriteria;

  SortCriteria m_sortCriteria;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_filterCriteriaHasBeenSet = false;
  bool m_sortCriteriaHasBeenSet = false;
};

}  // namespace Model
}  // namespace Inspector2
}  // namespace Aws
