﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/elasticache/ElastiCache_EXPORTS.h>
#include <aws/elasticache/model/RecurringCharge.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace ElastiCache {
namespace Model {

/**
 * <p>Represents the output of a <code>PurchaseReservedCacheNodesOffering</code>
 * operation.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/elasticache-2015-02-02/ReservedCacheNode">AWS
 * API Reference</a></p>
 */
class ReservedCacheNode {
 public:
  AWS_ELASTICACHE_API ReservedCacheNode() = default;
  AWS_ELASTICACHE_API ReservedCacheNode(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_ELASTICACHE_API ReservedCacheNode& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_ELASTICACHE_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_ELASTICACHE_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>The unique identifier for the reservation.</p>
   */
  inline const Aws::String& GetReservedCacheNodeId() const { return m_reservedCacheNodeId; }
  inline bool ReservedCacheNodeIdHasBeenSet() const { return m_reservedCacheNodeIdHasBeenSet; }
  template <typename ReservedCacheNodeIdT = Aws::String>
  void SetReservedCacheNodeId(ReservedCacheNodeIdT&& value) {
    m_reservedCacheNodeIdHasBeenSet = true;
    m_reservedCacheNodeId = std::forward<ReservedCacheNodeIdT>(value);
  }
  template <typename ReservedCacheNodeIdT = Aws::String>
  ReservedCacheNode& WithReservedCacheNodeId(ReservedCacheNodeIdT&& value) {
    SetReservedCacheNodeId(std::forward<ReservedCacheNodeIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The offering identifier.</p>
   */
  inline const Aws::String& GetReservedCacheNodesOfferingId() const { return m_reservedCacheNodesOfferingId; }
  inline bool ReservedCacheNodesOfferingIdHasBeenSet() const { return m_reservedCacheNodesOfferingIdHasBeenSet; }
  template <typename ReservedCacheNodesOfferingIdT = Aws::String>
  void SetReservedCacheNodesOfferingId(ReservedCacheNodesOfferingIdT&& value) {
    m_reservedCacheNodesOfferingIdHasBeenSet = true;
    m_reservedCacheNodesOfferingId = std::forward<ReservedCacheNodesOfferingIdT>(value);
  }
  template <typename ReservedCacheNodesOfferingIdT = Aws::String>
  ReservedCacheNode& WithReservedCacheNodesOfferingId(ReservedCacheNodesOfferingIdT&& value) {
    SetReservedCacheNodesOfferingId(std::forward<ReservedCacheNodesOfferingIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The cache node type for the reserved cache nodes.</p> <p>The following node
   * types are supported by ElastiCache. Generally speaking, the current generation
   * types provide more memory and computational power at lower cost when compared to
   * their equivalent previous generation counterparts.</p> <ul> <li> <p>General
   * purpose:</p> <ul> <li> <p>Current generation: </p> <p> <b>M7g node types</b>:
   * <code>cache.m7g.large</code>, <code>cache.m7g.xlarge</code>,
   * <code>cache.m7g.2xlarge</code>, <code>cache.m7g.4xlarge</code>,
   * <code>cache.m7g.8xlarge</code>, <code>cache.m7g.12xlarge</code>,
   * <code>cache.m7g.16xlarge</code> </p>  <p>For region availability, see <a
   * href="https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/CacheNodes.SupportedTypes.html#CacheNodes.SupportedTypesByRegion">Supported
   * Node Types</a> </p>  <p> <b>M6g node types</b> (available only for Redis
   * OSS engine version 5.0.6 onward and for Memcached engine version 1.5.16 onward):
   * <code>cache.m6g.large</code>, <code>cache.m6g.xlarge</code>,
   * <code>cache.m6g.2xlarge</code>, <code>cache.m6g.4xlarge</code>,
   * <code>cache.m6g.8xlarge</code>, <code>cache.m6g.12xlarge</code>,
   * <code>cache.m6g.16xlarge</code> </p> <p> <b>M5 node types:</b>
   * <code>cache.m5.large</code>, <code>cache.m5.xlarge</code>,
   * <code>cache.m5.2xlarge</code>, <code>cache.m5.4xlarge</code>,
   * <code>cache.m5.12xlarge</code>, <code>cache.m5.24xlarge</code> </p> <p> <b>M4
   * node types:</b> <code>cache.m4.large</code>, <code>cache.m4.xlarge</code>,
   * <code>cache.m4.2xlarge</code>, <code>cache.m4.4xlarge</code>,
   * <code>cache.m4.10xlarge</code> </p> <p> <b>T4g node types</b> (available only
   * for Redis OSS engine version 5.0.6 onward and Memcached engine version 1.5.16
   * onward): <code>cache.t4g.micro</code>, <code>cache.t4g.small</code>,
   * <code>cache.t4g.medium</code> </p> <p> <b>T3 node types:</b>
   * <code>cache.t3.micro</code>, <code>cache.t3.small</code>,
   * <code>cache.t3.medium</code> </p> <p> <b>T2 node types:</b>
   * <code>cache.t2.micro</code>, <code>cache.t2.small</code>,
   * <code>cache.t2.medium</code> </p> </li> <li> <p>Previous generation: (not
   * recommended. Existing clusters are still supported but creation of new clusters
   * is not supported for these types.)</p> <p> <b>T1 node types:</b>
   * <code>cache.t1.micro</code> </p> <p> <b>M1 node types:</b>
   * <code>cache.m1.small</code>, <code>cache.m1.medium</code>,
   * <code>cache.m1.large</code>, <code>cache.m1.xlarge</code> </p> <p> <b>M3 node
   * types:</b> <code>cache.m3.medium</code>, <code>cache.m3.large</code>,
   * <code>cache.m3.xlarge</code>, <code>cache.m3.2xlarge</code> </p> </li> </ul>
   * </li> <li> <p>Compute optimized:</p> <ul> <li> <p>Previous generation: (not
   * recommended. Existing clusters are still supported but creation of new clusters
   * is not supported for these types.)</p> <p> <b>C1 node types:</b>
   * <code>cache.c1.xlarge</code> </p> </li> </ul> </li> <li> <p>Memory
   * optimized:</p> <ul> <li> <p>Current generation: </p> <p> <b>R7g node types</b>:
   * <code>cache.r7g.large</code>, <code>cache.r7g.xlarge</code>,
   * <code>cache.r7g.2xlarge</code>, <code>cache.r7g.4xlarge</code>,
   * <code>cache.r7g.8xlarge</code>, <code>cache.r7g.12xlarge</code>,
   * <code>cache.r7g.16xlarge</code> </p>  <p>For region availability, see <a
   * href="https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/CacheNodes.SupportedTypes.html#CacheNodes.SupportedTypesByRegion">Supported
   * Node Types</a> </p>  <p> <b>R6g node types</b> (available only for Redis
   * OSS engine version 5.0.6 onward and for Memcached engine version 1.5.16 onward):
   * <code>cache.r6g.large</code>, <code>cache.r6g.xlarge</code>,
   * <code>cache.r6g.2xlarge</code>, <code>cache.r6g.4xlarge</code>,
   * <code>cache.r6g.8xlarge</code>, <code>cache.r6g.12xlarge</code>,
   * <code>cache.r6g.16xlarge</code> </p> <p> <b>R5 node types:</b>
   * <code>cache.r5.large</code>, <code>cache.r5.xlarge</code>,
   * <code>cache.r5.2xlarge</code>, <code>cache.r5.4xlarge</code>,
   * <code>cache.r5.12xlarge</code>, <code>cache.r5.24xlarge</code> </p> <p> <b>R4
   * node types:</b> <code>cache.r4.large</code>, <code>cache.r4.xlarge</code>,
   * <code>cache.r4.2xlarge</code>, <code>cache.r4.4xlarge</code>,
   * <code>cache.r4.8xlarge</code>, <code>cache.r4.16xlarge</code> </p> </li> <li>
   * <p>Previous generation: (not recommended. Existing clusters are still supported
   * but creation of new clusters is not supported for these types.)</p> <p> <b>M2
   * node types:</b> <code>cache.m2.xlarge</code>, <code>cache.m2.2xlarge</code>,
   * <code>cache.m2.4xlarge</code> </p> <p> <b>R3 node types:</b>
   * <code>cache.r3.large</code>, <code>cache.r3.xlarge</code>,
   * <code>cache.r3.2xlarge</code>, <code>cache.r3.4xlarge</code>,
   * <code>cache.r3.8xlarge</code> </p> </li> </ul> </li> </ul> <p> <b>Additional
   * node type info</b> </p> <ul> <li> <p>All current generation instance types are
   * created in Amazon VPC by default.</p> </li> <li> <p>Valkey or Redis OSS
   * append-only files (AOF) are not supported for T1 or T2 instances.</p> </li> <li>
   * <p>Valkey or Redis OSS Multi-AZ with automatic failover is not supported on T1
   * instances.</p> </li> <li> <p>The configuration variables <code>appendonly</code>
   * and <code>appendfsync</code> are not supported on Valkey, or on Redis OSS
   * version 2.8.22 and later.</p> </li> </ul>
   */
  inline const Aws::String& GetCacheNodeType() const { return m_cacheNodeType; }
  inline bool CacheNodeTypeHasBeenSet() const { return m_cacheNodeTypeHasBeenSet; }
  template <typename CacheNodeTypeT = Aws::String>
  void SetCacheNodeType(CacheNodeTypeT&& value) {
    m_cacheNodeTypeHasBeenSet = true;
    m_cacheNodeType = std::forward<CacheNodeTypeT>(value);
  }
  template <typename CacheNodeTypeT = Aws::String>
  ReservedCacheNode& WithCacheNodeType(CacheNodeTypeT&& value) {
    SetCacheNodeType(std::forward<CacheNodeTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time the reservation started.</p>
   */
  inline const Aws::Utils::DateTime& GetStartTime() const { return m_startTime; }
  inline bool StartTimeHasBeenSet() const { return m_startTimeHasBeenSet; }
  template <typename StartTimeT = Aws::Utils::DateTime>
  void SetStartTime(StartTimeT&& value) {
    m_startTimeHasBeenSet = true;
    m_startTime = std::forward<StartTimeT>(value);
  }
  template <typename StartTimeT = Aws::Utils::DateTime>
  ReservedCacheNode& WithStartTime(StartTimeT&& value) {
    SetStartTime(std::forward<StartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The duration of the reservation in seconds.</p>
   */
  inline int GetDuration() const { return m_duration; }
  inline bool DurationHasBeenSet() const { return m_durationHasBeenSet; }
  inline void SetDuration(int value) {
    m_durationHasBeenSet = true;
    m_duration = value;
  }
  inline ReservedCacheNode& WithDuration(int value) {
    SetDuration(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The fixed price charged for this reserved cache node.</p>
   */
  inline double GetFixedPrice() const { return m_fixedPrice; }
  inline bool FixedPriceHasBeenSet() const { return m_fixedPriceHasBeenSet; }
  inline void SetFixedPrice(double value) {
    m_fixedPriceHasBeenSet = true;
    m_fixedPrice = value;
  }
  inline ReservedCacheNode& WithFixedPrice(double value) {
    SetFixedPrice(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The hourly price charged for this reserved cache node.</p>
   */
  inline double GetUsagePrice() const { return m_usagePrice; }
  inline bool UsagePriceHasBeenSet() const { return m_usagePriceHasBeenSet; }
  inline void SetUsagePrice(double value) {
    m_usagePriceHasBeenSet = true;
    m_usagePrice = value;
  }
  inline ReservedCacheNode& WithUsagePrice(double value) {
    SetUsagePrice(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of cache nodes that have been reserved.</p>
   */
  inline int GetCacheNodeCount() const { return m_cacheNodeCount; }
  inline bool CacheNodeCountHasBeenSet() const { return m_cacheNodeCountHasBeenSet; }
  inline void SetCacheNodeCount(int value) {
    m_cacheNodeCountHasBeenSet = true;
    m_cacheNodeCount = value;
  }
  inline ReservedCacheNode& WithCacheNodeCount(int value) {
    SetCacheNodeCount(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the reserved cache node.</p>
   */
  inline const Aws::String& GetProductDescription() const { return m_productDescription; }
  inline bool ProductDescriptionHasBeenSet() const { return m_productDescriptionHasBeenSet; }
  template <typename ProductDescriptionT = Aws::String>
  void SetProductDescription(ProductDescriptionT&& value) {
    m_productDescriptionHasBeenSet = true;
    m_productDescription = std::forward<ProductDescriptionT>(value);
  }
  template <typename ProductDescriptionT = Aws::String>
  ReservedCacheNode& WithProductDescription(ProductDescriptionT&& value) {
    SetProductDescription(std::forward<ProductDescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The offering type of this reserved cache node.</p>
   */
  inline const Aws::String& GetOfferingType() const { return m_offeringType; }
  inline bool OfferingTypeHasBeenSet() const { return m_offeringTypeHasBeenSet; }
  template <typename OfferingTypeT = Aws::String>
  void SetOfferingType(OfferingTypeT&& value) {
    m_offeringTypeHasBeenSet = true;
    m_offeringType = std::forward<OfferingTypeT>(value);
  }
  template <typename OfferingTypeT = Aws::String>
  ReservedCacheNode& WithOfferingType(OfferingTypeT&& value) {
    SetOfferingType(std::forward<OfferingTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The state of the reserved cache node.</p>
   */
  inline const Aws::String& GetState() const { return m_state; }
  inline bool StateHasBeenSet() const { return m_stateHasBeenSet; }
  template <typename StateT = Aws::String>
  void SetState(StateT&& value) {
    m_stateHasBeenSet = true;
    m_state = std::forward<StateT>(value);
  }
  template <typename StateT = Aws::String>
  ReservedCacheNode& WithState(StateT&& value) {
    SetState(std::forward<StateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The recurring price charged to run this reserved cache node.</p>
   */
  inline const Aws::Vector<RecurringCharge>& GetRecurringCharges() const { return m_recurringCharges; }
  inline bool RecurringChargesHasBeenSet() const { return m_recurringChargesHasBeenSet; }
  template <typename RecurringChargesT = Aws::Vector<RecurringCharge>>
  void SetRecurringCharges(RecurringChargesT&& value) {
    m_recurringChargesHasBeenSet = true;
    m_recurringCharges = std::forward<RecurringChargesT>(value);
  }
  template <typename RecurringChargesT = Aws::Vector<RecurringCharge>>
  ReservedCacheNode& WithRecurringCharges(RecurringChargesT&& value) {
    SetRecurringCharges(std::forward<RecurringChargesT>(value));
    return *this;
  }
  template <typename RecurringChargesT = RecurringCharge>
  ReservedCacheNode& AddRecurringCharges(RecurringChargesT&& value) {
    m_recurringChargesHasBeenSet = true;
    m_recurringCharges.emplace_back(std::forward<RecurringChargesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the reserved cache node.</p> <p>Example:
   * <code>arn:aws:elasticache:us-east-1:123456789012:reserved-instance:ri-2017-03-27-08-33-25-582</code>
   * </p>
   */
  inline const Aws::String& GetReservationARN() const { return m_reservationARN; }
  inline bool ReservationARNHasBeenSet() const { return m_reservationARNHasBeenSet; }
  template <typename ReservationARNT = Aws::String>
  void SetReservationARN(ReservationARNT&& value) {
    m_reservationARNHasBeenSet = true;
    m_reservationARN = std::forward<ReservationARNT>(value);
  }
  template <typename ReservationARNT = Aws::String>
  ReservedCacheNode& WithReservationARN(ReservationARNT&& value) {
    SetReservationARN(std::forward<ReservationARNT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_reservedCacheNodeId;

  Aws::String m_reservedCacheNodesOfferingId;

  Aws::String m_cacheNodeType;

  Aws::Utils::DateTime m_startTime{};

  int m_duration{0};

  double m_fixedPrice{0.0};

  double m_usagePrice{0.0};

  int m_cacheNodeCount{0};

  Aws::String m_productDescription;

  Aws::String m_offeringType;

  Aws::String m_state;

  Aws::Vector<RecurringCharge> m_recurringCharges;

  Aws::String m_reservationARN;
  bool m_reservedCacheNodeIdHasBeenSet = false;
  bool m_reservedCacheNodesOfferingIdHasBeenSet = false;
  bool m_cacheNodeTypeHasBeenSet = false;
  bool m_startTimeHasBeenSet = false;
  bool m_durationHasBeenSet = false;
  bool m_fixedPriceHasBeenSet = false;
  bool m_usagePriceHasBeenSet = false;
  bool m_cacheNodeCountHasBeenSet = false;
  bool m_productDescriptionHasBeenSet = false;
  bool m_offeringTypeHasBeenSet = false;
  bool m_stateHasBeenSet = false;
  bool m_recurringChargesHasBeenSet = false;
  bool m_reservationARNHasBeenSet = false;
};

}  // namespace Model
}  // namespace ElastiCache
}  // namespace Aws
