/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.qpid.server.security;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertInstanceOf;
import static org.junit.jupiter.api.Assertions.assertNotEquals;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertTrue;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.nio.charset.StandardCharsets;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.interfaces.RSAPublicKey;
import java.util.Base64;
import java.util.List;
import java.util.Map;

import javax.net.ssl.KeyManager;

import org.junit.jupiter.api.Test;

import org.apache.qpid.server.model.Broker;
import org.apache.qpid.server.model.BrokerModel;
import org.apache.qpid.server.model.BrokerTestHelper;
import org.apache.qpid.server.model.ConfiguredObjectFactory;
import org.apache.qpid.server.model.KeyStore;
import org.apache.qpid.test.utils.UnitTestBase;

class AutoGeneratedSelfSignedKeyStoreTest extends UnitTestBase
{
    private static final Broker<?> BROKER = BrokerTestHelper.createBrokerMock();
    private static final ConfiguredObjectFactory FACTORY = BrokerModel.getInstance().getObjectFactory();
    private static final String NAME = "myKeyStore";

    private final AutoGeneratedSelfSignedKeyStore<?> _keyStore = createAutoGeneratedSelfSignedKeyStore();

    @Test
    void creation() throws Exception
    {
        final KeyManager[] keyManager = _keyStore.getKeyManagers();

        assertNotNull(keyManager);
        assertEquals(1, keyManager.length, "Unexpected number of key managers");
        assertNotNull(keyManager[0], "Key manager unexpected null");
    }

    @Test
    void regenerate()
    {
        final String privateKey = _keyStore.getEncodedPrivateKey();
        final String certificate = _keyStore.getEncodedCertificate();

        assertNotNull(privateKey);
        assertNotNull(certificate);

        _keyStore.regenerateCertificate();

        final String regeneratedPrivateKey = _keyStore.getEncodedPrivateKey();
        final String regeneratedCertificate = _keyStore.getEncodedCertificate();

        assertNotNull(regeneratedPrivateKey);
        assertNotNull(regeneratedCertificate);

        assertNotEquals(privateKey, regeneratedPrivateKey, "Regenerated private key shouldn't be equal to the original private key");
        assertNotEquals(certificate, regeneratedCertificate, "Regenerated certificate shouldn't be equal to the original certificate");
    }

    @Test
    void privateKeyEntryCertificate()
    {
        final List<CertificateDetails> certificateDetails = _keyStore.getCertificateDetails();

        assertEquals(1, certificateDetails.size(), "Unexpected number of certificates");
        assertEquals("myKeyStore", certificateDetails.get(0).getAlias(), "Unexpected alias name");
    }

    @Test
    void content() throws Exception
    {
        try (final ByteArrayOutputStream baos = new ByteArrayOutputStream())
        {
            _keyStore.getCertificate().write(baos);

            final CertificateFactory certFactory = CertificateFactory.getInstance("X.509");
            final X509Certificate certificate = (X509Certificate) certFactory
                    .generateCertificate(new ByteArrayInputStream(baos.toByteArray()));
            final String encodedCertificate = new String(Base64.getEncoder().encode(certificate.getEncoded()), StandardCharsets.UTF_8);

            assertEquals("X.509", certificate.getType(), "Certificate type mismatch");
            assertTrue(_keyStore.getSignatureAlgorithm().equalsIgnoreCase(certificate.getSigAlgName()),
                       "Certificate signature algorithm mismatch");
            assertInstanceOf(RSAPublicKey.class, certificate.getPublicKey(), "Key class mismatch");
            assertEquals(_keyStore.getKeyLength(), ((RSAPublicKey) certificate.getPublicKey()).getModulus().bitLength(), "Key length mismatch");
            assertEquals(_keyStore.getEncodedCertificate(), encodedCertificate, "Certificate content mismatch");
        }
    }

    private AutoGeneratedSelfSignedKeyStore<?> createAutoGeneratedSelfSignedKeyStore()
    {
        final Map<String, Object> attributes = Map.of(AutoGeneratedSelfSignedKeyStore.NAME, NAME,
                AutoGeneratedSelfSignedKeyStore.TYPE, "AutoGeneratedSelfSigned");
        return (AutoGeneratedSelfSignedKeyStore<?>) FACTORY.create(KeyStore.class, attributes, BROKER);
    }
}
