/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.s3.analyticsaccelerator.io.physical.data;

import java.io.Closeable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import lombok.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.s3.analyticsaccelerator.common.Metrics;
import software.amazon.s3.analyticsaccelerator.common.Preconditions;
import software.amazon.s3.analyticsaccelerator.io.physical.PhysicalIOConfiguration;
import software.amazon.s3.analyticsaccelerator.io.physical.data.BlobStoreIndexCache;
import software.amazon.s3.analyticsaccelerator.io.physical.data.Block;
import software.amazon.s3.analyticsaccelerator.request.Range;
import software.amazon.s3.analyticsaccelerator.util.BlockKey;
import software.amazon.s3.analyticsaccelerator.util.MetricKey;

public class BlockStore
implements Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(BlockStore.class);
    private final BlobStoreIndexCache indexCache;
    private final Metrics aggregatingMetrics;
    private final PhysicalIOConfiguration configuration;
    private final Map<Integer, Block> blocks;

    public BlockStore(@NonNull BlobStoreIndexCache indexCache, @NonNull Metrics aggregatingMetrics, @NonNull PhysicalIOConfiguration configuration) {
        if (indexCache == null) {
            throw new NullPointerException("indexCache is marked non-null but is null");
        }
        if (aggregatingMetrics == null) {
            throw new NullPointerException("aggregatingMetrics is marked non-null but is null");
        }
        if (configuration == null) {
            throw new NullPointerException("configuration is marked non-null but is null");
        }
        this.indexCache = indexCache;
        this.aggregatingMetrics = aggregatingMetrics;
        this.configuration = configuration;
        this.blocks = new HashMap<Integer, Block>();
    }

    public Optional<Block> getBlock(long pos) {
        Preconditions.checkArgument(0L <= pos, "`pos` must not be negative");
        return this.getBlockByIndex(this.getPositionIndex(pos));
    }

    public Optional<Block> getBlockByIndex(int index) {
        Preconditions.checkArgument(0 <= index, "`index` must not be negative");
        return Optional.ofNullable(this.blocks.get(index));
    }

    public void add(Block block) {
        int blockIndex = this.getBlockIndex(block);
        if (this.blocks.containsKey(blockIndex)) {
            LOG.debug("Block already exists at index {}, skipping add", (Object)blockIndex);
        } else {
            this.blocks.put(blockIndex, block);
        }
    }

    public void remove(Block block) {
        if (block == null) {
            return;
        }
        int blockIndex = this.getBlockIndex(block);
        if (this.blocks.remove(blockIndex) != null && block.isDataReady()) {
            this.aggregatingMetrics.reduce(MetricKey.MEMORY_USAGE, block.getLength());
            this.safeClose(block);
        }
    }

    public List<Integer> getMissingBlockIndexesInRange(Range range) {
        return this.getMissingBlockIndexesInRange(this.getPositionIndex(range.getStart()), this.getPositionIndex(range.getEnd()));
    }

    private List<Integer> getMissingBlockIndexesInRange(int startIndex, int endIndex) {
        ArrayList<Integer> missingBlockIndexes = new ArrayList<Integer>();
        for (int i = startIndex; i <= endIndex; ++i) {
            if (this.blocks.containsKey(i)) continue;
            missingBlockIndexes.add(i);
        }
        return missingBlockIndexes;
    }

    public void cleanUp() {
        Iterator<Map.Entry<Integer, Block>> iterator = this.blocks.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry<Integer, Block> entry = iterator.next();
            Block block = entry.getValue();
            BlockKey blockKey = block.getBlockKey();
            if (!block.isDataReady() || this.indexCache.contains(blockKey)) continue;
            try {
                iterator.remove();
                this.aggregatingMetrics.reduce(MetricKey.MEMORY_USAGE, blockKey.getRange().getLength());
            }
            catch (Exception e) {
                LOG.error("Error in removing block {}", (Object)e.getMessage());
            }
        }
    }

    private int getBlockIndex(Block block) {
        return this.getPositionIndex(block.getBlockKey().getRange().getStart());
    }

    private int getPositionIndex(long pos) {
        return (int)(pos / this.configuration.getReadBufferSize());
    }

    @Override
    public void close() {
        for (Block block : this.blocks.values()) {
            this.safeClose(block);
        }
        this.blocks.clear();
    }

    private void safeClose(Block block) {
        try {
            block.close();
        }
        catch (Exception e) {
            LOG.error("Exception when closing Block in the BlockStore", (Throwable)e);
        }
    }

    public boolean isEmpty() {
        return this.blocks.isEmpty();
    }
}

